import json
from tqdm import tqdm
import torch
from diffusers import FluxPipeline
from huggingface_hub import login
import os

# Set your dataset path and annotation file path
data_dir = r"path_to_save_images"
json_file = r"path_to_annotation.json"

def generate_image_with_flux(pipe, prompt):
    """
    Generate an image using the Flux model.

    Args:
        pipe (FluxPipeline): The loaded Flux model pipeline.
        prompt (str): The text prompt used to generate the image.

    Returns:
        PIL.Image: The generated image.
    """
    image = pipe(
        prompt,
        height=1024,
        width=1024,
        guidance_scale=3.5,
        num_inference_steps=50,
        max_sequence_length=512,
        generator=torch.Generator("cpu").manual_seed(0)
    ).images[0]
    return image

def read_json_file(file_path):
    """
    Read and parse a JSON file.

    Args:
        file_path (str): Path to the JSON file.

    Returns:
        dict or list: Parsed JSON content.
    """
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            return json.load(f)
    except FileNotFoundError:
        print(f"Error: File not found: {file_path}")
    except json.JSONDecodeError as e:
        print(f"Error: Failed to parse JSON file - {e}")
    return None

def main():
    # Load annotations
    annotation = read_json_file(json_file)

    # Authenticate with Hugging Face (replace with your own method or env variable)
    login(token="your_huggingface_token_here")

    # Load the Flux model pipeline
    pipe = FluxPipeline.from_pretrained(
        "black-forest-labs/FLUX.1-dev",
        torch_dtype=torch.bfloat16,
        cache_dir=r"path_to_cache_dir"
    )
    pipe.enable_model_cpu_offload()

    # Select a portion of the dataset
    img_id_list = list(annotation.keys())[640:]

    for img_id in tqdm(img_id_list):
        caption = annotation[img_id]["caption"]
        img = generate_image_with_flux(pipe, caption)
        save_path = os.path.join(data_dir, f"{img_id}.png")
        img.save(save_path)

if __name__ == "__main__":
    main()
