import json
from nltk.translate.bleu_score import sentence_bleu, SmoothingFunction
from nltk.translate.meteor_score import meteor_score
from rouge_score import rouge_scorer
from bert_score import score
import nltk
# nltk.download('punkt_tab')
# nltk.download('wordnet')
# nltk.download('punkt')

from transformers import logging
logging.set_verbosity_error()

from tqdm import tqdm
from nltk.tokenize import word_tokenize

def read_log_file(file_path, encoding='utf-8'):
    try:
        with open(file_path, 'r', encoding=encoding) as file:
            return file.read()
    except FileNotFoundError:
        print(f"Error: File '{file_path}' not found")
        return None
    except PermissionError:
        print(f"Error: Permission denied to read file '{file_path}'")
        return None
    except UnicodeDecodeError:
        print(f"Error: Decoding error using {encoding} encoding")
        return None
    except Exception as e:
        print(f"Unknown error: {e}")
        return None

def load_img_labels(json_file_path):
    with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
        return json.load(jsonfile)

def evaluate_captions(predicted: str, reference: str, performance: dict):
    P, R, F1 = score([predicted], [reference], lang='en', verbose=False)
    performance["BERTScore"]["Precision"].append(P.item())
    performance["BERTScore"]["Recall"].append(R.item())
    performance["BERTScore"]["F1"].append(F1.item())

    ref_tokens = reference.split()
    pred_tokens = predicted.split()
    smoothie = SmoothingFunction().method4
    bleu1 = sentence_bleu([ref_tokens], pred_tokens, weights=(1, 0, 0, 0), smoothing_function=smoothie)
    bleu2 = sentence_bleu([ref_tokens], pred_tokens, weights=(0.5, 0.5, 0, 0), smoothing_function=smoothie)
    bleu4 = sentence_bleu([ref_tokens], pred_tokens, weights=(0.25, 0.25, 0.25, 0.25), smoothing_function=smoothie)

    performance["BLEU"]["BLEU-1"].append(bleu1)
    performance["BLEU"]["BLEU-2"].append(bleu2)
    performance["BLEU"]["BLEU-4"].append(bleu4)

    meteor = meteor_score([word_tokenize(reference)], word_tokenize(predicted))
    performance["METEOR"].append(meteor)

    scorer = rouge_scorer.RougeScorer(['rouge1', 'rouge2', 'rougeL'], use_stemmer=True)
    scores = scorer.score(reference, predicted)
    for key in scores:
        performance["ROUGE"][key.upper()]["Precision"].append(scores[key].precision)
        performance["ROUGE"][key.upper()]["Recall"].append(scores[key].recall)
        performance["ROUGE"][key.upper()]["F1"].append(scores[key].fmeasure)

def read_text_file(file_path, encoding='utf-8', strip_lines=False, as_list=False):
    try:
        with open(file_path, 'r', encoding=encoding) as file:
            if as_list:
                lines = file.readlines()
                return [line.strip() for line in lines] if strip_lines else lines
            else:
                content = file.read()
                return content.strip() if strip_lines else content
    except FileNotFoundError:
        print(f"Error: File '{file_path}' not found")
        return None
    except PermissionError:
        print(f"Error: Permission denied to read file '{file_path}'")
        return None
    except UnicodeDecodeError:
        print(f"Error: Decoding error using {encoding} encoding")
        return None
    except Exception as e:
        print(f"Unknown error: {e}")
        return None

def main():
    label_json_path = '/path/to/contrasense_QA.json'
    label_json = load_img_labels(label_json_path)

    performance = {
        "BERTScore": {"Precision": [], "Recall": [], "F1": []},
        "BLEU": {"BLEU-1": [], "BLEU-2": [], "BLEU-4": []},
        "METEOR": [],
        "ROUGE": {
            "ROUGE1": {"Precision": [], "Recall": [], "F1": []},
            "ROUGE2": {"Precision": [], "Recall": [], "F1": []},
            "ROUGEL": {"Precision": [], "Recall": [], "F1": []}
        }
    }

    log_file_path = '/path/to/model_log_file.log'
    log_file = read_log_file(log_file_path)
    log_lines = log_file.split("\n")
    answer_dict = eval(log_lines[-3][len("2025-05-27 07:01:49,692 - INFO - "):])

    for img_id in tqdm(answer_dict.keys()):
        reference_caption = label_json[img_id]['caption']
        predicted_caption = answer_dict[img_id]['caption_answer']
        if isinstance(predicted_caption, list):
            predicted_caption = predicted_caption[0]
        evaluate_captions(predicted_caption, reference_caption, performance)

    print("BERTScore:")
    print(f"  Precision: {sum(performance['BERTScore']['Precision']) / len(performance['BERTScore']['Precision']):.4f}")
    print(f"  Recall:    {sum(performance['BERTScore']['Recall']) / len(performance['BERTScore']['Recall']):.4f}")
    print(f"  F1:        {sum(performance['BERTScore']['F1']) / len(performance['BERTScore']['F1']):.4f}\n")

    print("BLEU:")
    print(f"  BLEU-1: {sum(performance['BLEU']['BLEU-1']) / len(performance['BLEU']['BLEU-1']):.4f}")
    print(f"  BLEU-2: {sum(performance['BLEU']['BLEU-2']) / len(performance['BLEU']['BLEU-2']):.4f}")
    print(f"  BLEU-4: {sum(performance['BLEU']['BLEU-4']) / len(performance['BLEU']['BLEU-4']):.4f}\n")

    print(f"METEOR: {sum(performance['METEOR']) / len(performance['METEOR']):.4f}\n")

    print("ROUGE:")
    for key in performance["ROUGE"].keys():
        print(f"  {key.upper()}: Precision={sum(performance['ROUGE'][key]['Precision']) / len(performance['ROUGE'][key]['Precision']):.4f}, "
              f"Recall={sum(performance['ROUGE'][key]['Recall']) / len(performance['ROUGE'][key]['Recall']):.4f}, "
              f"F1={sum(performance['ROUGE'][key]['F1']) / len(performance['ROUGE'][key]['F1']):.4f}")

    print(f"Total samples evaluated: {len(performance['METEOR'])}")

if __name__ == "__main__":
    main()
