import torch
import time
from tqdm import tqdm
import csv
import logging
import os, sys
import copy
import torch.nn as nn
from torch.utils.data import Dataset
import torch.distributed as dist
from typing import List, Dict, Tuple, Any
import json
from PIL import Image

from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary,
    extend_merge_results as extend_merge_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )

from transformers import AutoTokenizer, AutoProcessor
from qwen_vl_utils import process_vision_info


from torch.distributed._tensor import DTensor

def convert_dtensor_to_tensor(module: nn.Module) -> None:
    """
    Recursively converts all DTensor parameters and buffers in a module to regular Tensors.
    
    This function ensures that all DTensor objects are replaced with their local tensor copies,
    making the module compatible with standard PyTorch operations.
    
    Args:
        module (nn.Module): The PyTorch module whose DTensor parameters and buffers will be converted.
    
    Returns:
        None
    """
    # Convert DTensor parameters to standard Tensors
    for name, param in module.named_parameters(recurse=False):
        if isinstance(param, DTensor):
            local_tensor = param._local_tensor.detach().clone()
            new_param = nn.Parameter(local_tensor, requires_grad=param.requires_grad)
            module.register_parameter(name, new_param)
    
    # Convert DTensor buffers to standard Tensors
    for name, buffer in module.named_buffers(recurse=False):
        if isinstance(buffer, DTensor):
            local_tensor = buffer._local_tensor.detach().clone()
            module.register_buffer(name, local_tensor)
    
    # Recursively process child modules
    for child_module in module.children():
        convert_dtensor_to_tensor(child_module)

def qwen_vl_collate_fn(batch: List[Dict[str, Any]]) -> Dict[str, List[Any]]:
    """
    Custom collate function for processing a batch of samples for Qwen-VL.
    
    This function organizes batch data into lists grouped by their corresponding keys.
    
    Args:
        batch (List[Dict[str, Any]]): A batch of samples where each sample is a dictionary
            containing various keys related to Qwen-VL inputs and labels.
    
    Returns:
        Dict[str, List[Any]]: A dictionary where each key corresponds to a list of values
            from the batch, ensuring structured batching for model input.
    """

    metadata = {
        'img_id'                        : [item['img_id'] for item in batch],
        'img_path'                      : [item['img_path'] for item in batch],
        'caption'                       : [item['caption'] for item in batch],
        'object_class_QA'               : [item['object_class_QA'] for item in batch],
        'object_colour_QA'              : [item['object_colour_QA'] for item in batch],
        'object_num_QA'                 : [item['object_num_QA'] for item in batch],
        'is_counterintuitive_QA'        : [item['is_counterintuitive_QA'] for item in batch],
        'relation_QA'                   : [item['relation_QA'] for item in batch],
        'Anomaly_Recognition_QA'        : [item['Anomaly_Recognition_QA'] for item in batch]
    }

    return {
        **metadata
    }


class QwenVLSeriesDataset(Dataset):
    """
    A PyTorch Dataset for processing Qwen-VL image-category pairs with presence and absence prompts.
    
    This dataset loads image-category-label pairs from a CSV file, generates corresponding presence and absence prompts,
    and structures the data for multimodal model input.
    """
    def __init__(self, args: Any) -> None:
        """
        Initializes the dataset with image-category pairs and paths.
        
        Args:
            args (Any): Configuration object containing dataset paths and templates.
        """
        self.img_category_pairs: List[Tuple[str, str, int]] = []  # Stores (img_id, category, label) tuples
        self.datasets_img_path: str = args.DATASET_PATH.get(args.dataset)
        self.args = args

        self._load_img_label(args.DATASET_LABEL.get(args.dataset))
        self._get_image_filenames(args.DATASET_PATH.get(args.dataset))
        
        # Default message structure template
        self.message = [
            {"role": "user", "content": [
                {"type": "image", "image": None},
                {"type": "text", "text": None}]}]

    def _load_img_label(self, json_file_path):
        """Loads labels from the json file."""
        with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
            self.contrasense_json = json.load(jsonfile)
            
    def _get_image_filenames(self, folder_path, image_extensions=None):
        if image_extensions is None:
            image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

        self.image_filenames = []
        for filename in os.listdir(folder_path):
            if os.path.isfile(os.path.join(folder_path, filename)):
                ext = os.path.splitext(filename)[1].lower()
                if ext in image_extensions:
                    self.image_filenames.append(filename)
        
        logging.info(f"Found {len(self.image_filenames)} image(s) in '{folder_path}'.")

    def _get_questions(self, questions):
        question_keys = questions.keys()
        for question_key in question_keys:
            if "object class Question" in question_key:
                object_class_QA = questions[question_key]
            elif "object colour Question" in question_key:
                object_colour_QA = questions[question_key]
            elif "object num Question" in question_key:
                object_num_QA = questions[question_key]
            elif "is counterintuitive Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "relation Question" in question_key:
                relation_QA = questions[question_key]
            elif "Anomaly Recognition Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
            elif "function Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "Commonsense Reasoning Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
        return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA

    def __len__(self):
        return len(self.image_filenames)

    def __getitem__(self, idx: int) -> Dict[str, Any]:
        """
        Retrieves the sample at the specified index.
        
        Args:
            idx (int): Index of the sample.
        
        Returns:
            Dict[str, Any]: A dictionary containing the processed sample with messages, labels, and prompts.
        """
        img_id = self.image_filenames[idx]
        # Determine image path
        img_path = os.path.join(self.datasets_img_path, img_id)
        
        if not os.path.exists(img_path):
            raise FileNotFoundError(f"Image file not found: {img_path}")
        
        img_id = img_id.split('.')[0]

        # logging.info(img_id)
        
        data = self.contrasense_json.get(img_id)
        
        caption = data["caption"]
        questions = data["questions"]

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = self._get_questions(questions)
 
        return {
            'img_id'                        : img_id,
            'img_path'                      : img_path,
            'caption'                       : caption,
            'object_class_QA'               : object_class_QA,
            'object_colour_QA'              : object_colour_QA,
            'object_num_QA'                 : object_num_QA,
            'is_counterintuitive_QA'        : is_counterintuitive_QA,
            'relation_QA'                   : relation_QA,
            'Anomaly_Recognition_QA'        : Anomaly_Recognition_QA
        }

def load_model_and_processor(args: Any, model_path: str) -> Tuple[Any, AutoProcessor]:
    """
    Loads the specified Qwen model and its corresponding processor.
    
    Args:
        args (Any): Configuration object containing model evaluation settings and cache paths.
        model_path (str): Path to the pre-trained model.
    
    Returns:
        Tuple[Any, AutoProcessor]: The loaded model and processor.
    """
    if args.eval_model == "Qwen2-VL-7B-Instruct":
        from transformers import Qwen2VLForConditionalGeneration
        model = Qwen2VLForConditionalGeneration.from_pretrained(
            model_path, 
            torch_dtype="auto", 
            cache_dir=args.model_para_path).eval().to(args.device)
        processor = AutoProcessor.from_pretrained(model_path, cache_dir=args.model_para_path)

    elif "Qwen2_5" in args.eval_model:
        from transformers import Qwen2_5_VLForConditionalGeneration
        # default: Load the model on the available device(s)
        model = Qwen2_5_VLForConditionalGeneration.from_pretrained(
            model_path,
            torch_dtype=torch.bfloat16,
            attn_implementation="flash_attention_2",
            # device_map="auto",
            use_cache=False,    # Resolves flash_attn_2 issue
            cache_dir=args.model_para_path
        )

        # default processer
        processor = AutoProcessor.from_pretrained(
            model_path,
            use_cache=False,     # Resolves flash_attn_2 issue
            cache_dir=args.model_para_path
            )
    return model, processor

def construct_batch_data(
        args: Any, 
        messages_presence: List[Any], 
        messages_absence: List[Any], 
        presence_label: List[int], 
        absence_label: List[int], 
        processor: Any
    ) -> Tuple[Any, List[int]]:
    """
    Constructs batch data for model inference.
    
    Args:
        args (Any): Configuration object containing batch size and model settings.
        messages_presence (List[Any]): List of presence-related messages.
        messages_absence (List[Any]): List of absence-related messages.
        presence_label (List[int]): List of presence labels.
        absence_label (List[int]): List of absence labels.
        processor (Any): Processor for text and vision data.
    
    Returns:
        Tuple[Any, List[int]]: Processed input tensors and corresponding labels.
    """
    messages = [elem for idx in range(args.batchsize) for elem in (messages_presence[idx], messages_absence[idx])]
    labels = [elem for idx in range(args.batchsize) for elem in (presence_label[idx], absence_label[idx])]

    # Prepare text inputs for batch inference
    texts = [
        processor.apply_chat_template(msg, tokenize=False, add_generation_prompt=True)
        for msg in messages
    ]

    # Process vision-related inputs (images and videos)
    image_inputs, video_inputs = process_vision_info(messages)

    # Construct model inputs
    inputs = processor(
        text=texts,
        images=image_inputs,
        videos=video_inputs,
        padding=True,
        **({"padding_side": "left"} if "2_5" in args.eval_model else {}),
        return_tensors="pt",
    )

    inputs = inputs.to("cuda")

    return inputs, labels

def construct_qwen_data(args, processor, QA, img_path):

    if not os.path.exists(img_path):
        raise FileNotFoundError(f"Image file not found: {img_path}")

    try:
        if type(QA) == dict:
            question = "Please answer the following single-choice question by selecting the one correct option. Only output the letter of the correct answer, with no explanation." + "\n" + QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4] + "\n" + "Answer: "
        elif type(QA) == str:
            question = QA
    except:
        logging.info(QA)
        exit()
    
    image = Image.open(img_path).convert("RGB")
    image = image.resize((args.img_size, args.img_size), Image.BICUBIC)

    messages = [
        {
            "role": "user",
            "content": [
                {
                    "type": "image",
                    "image": image,
                },
                {"type": "text", "text": question},
            ],
        }
    ]


    # Preparation for inference
    text = processor.apply_chat_template(
        messages, tokenize=False, add_generation_prompt=True
    )
    image_inputs, video_inputs = process_vision_info(messages)
    inputs = processor(
        text=[text],
        images=image_inputs,
        videos=video_inputs,
        padding=True,
        return_tensors="pt",
    )
    inputs = inputs.to("cuda")

    return inputs

def qwenvl_forward(
    args: Any, 
    inputs: Any, 
    model: Any, 
    processor: Any
) -> List[str]:
    """
    Performs forward inference using the Qwen-VL model.
    
    Args:
        args (Any): Configuration object containing model settings.
        inputs (Any): Preprocessed input tensors for the model.
        model (Any): Qwen-VL model instance.
        processor (Any): Processor used for decoding model outputs.
    
    Returns:
        List[str]: List of decoded output texts.
    """
    # Batch Inference
    generated_ids = model.module.generate(**inputs, use_cache=True, max_new_tokens=256, do_sample="2_5" not in args.eval_model)

    # Trim generated IDs based on input lengths
    generated_ids_trimmed = [
        out_ids[len(in_ids) :] for in_ids, out_ids in zip(inputs.input_ids, generated_ids)
    ]

    # Decode generated outputs into text
    output_texts = processor.batch_decode(
        generated_ids_trimmed, skip_special_tokens=True, clean_up_tokenization_spaces=False
    )
    return output_texts


def run_model(args):
    """
    Run the Qwen-VL Series model evaluation.
    
    Args:
        args: Argument parser containing configuration parameters.

    Returns:
        None
    """
    model_path = args.module

    # Load model and processor
    model, processor = load_model_and_processor(args, model_path)

    # Initialize dataset and dataloader
    dataset = QwenVLSeriesDataset(args)
    sampler = torch.utils.data.distributed.DistributedSampler(dataset, shuffle=False)
    data_loader = torch.utils.data.DataLoader(dataset, batch_size=args.batchsize, sampler=sampler, collate_fn=qwen_vl_collate_fn)

    # Enable Distributed Data Parallel (DDP)
    # convert_dtensor_to_tensor(model)
    model = model.to(f"cuda:{args.local_rank}")
    model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.local_rank])

    torch.cuda.empty_cache()
    start_time = time.time()

    # Initialize progress bar
    pbar = tqdm(total=len(data_loader), desc="Evaluating") if dist.get_rank() == 0 else None

    # Storage for predictions and labels
    all_data = {}

    for idx, batch_data in enumerate(data_loader):
        if pbar is not None:
            pbar.update(1)
            pbar.set_postfix_str(f"GPU: {dist.get_rank()}")

        # Load batch data
        img_id                              = batch_data.get('img_id')[0]
        img_path                            = batch_data.get('img_path')[0]
        caption                             = batch_data.get('caption')
        object_class_QA                     = batch_data.get('object_class_QA')[0]
        object_colour_QA                    = batch_data.get('object_colour_QA')[0]
        object_num_QA                       = batch_data.get('object_num_QA')[0]
        is_counterintuitive_QA              = batch_data.get('is_counterintuitive_QA')[0]
        relation_QA                         = batch_data.get('relation_QA')[0]
        Anomaly_Recognition_QA              = batch_data.get('Anomaly_Recognition_QA')[0]
        caption_instruction                 = args.caption_instruction
        # print(presence_question_template)

        # Construct batch input
        # class question & model inference
        object_class_input = construct_qwen_data(args, processor, object_class_QA, img_path)
        object_class_answer = qwenvl_forward(args, object_class_input, model, processor)

        # colour question & model inference
        object_colour_input = construct_qwen_data(args, processor, object_colour_QA, img_path)
        object_colour_answer = qwenvl_forward(args, object_colour_input, model, processor)

        # num question & model inference
        object_num_input = construct_qwen_data(args, processor, object_num_QA, img_path)
        object_num_answer = qwenvl_forward(args, object_num_input, model, processor)

        # counterintuitive question & model inference
        is_counterintuitive_input = construct_qwen_data(args, processor, is_counterintuitive_QA, img_path)
        counterintuitive_answer = qwenvl_forward(args, is_counterintuitive_input, model, processor)

        # relation question & model inference
        relation_input = construct_qwen_data(args, processor, relation_QA, img_path)
        relation_answer = qwenvl_forward(args, relation_input, model, processor)

        # anomaly recognition question & model inference
        Anomaly_Recognition_input = construct_qwen_data(args, processor, Anomaly_Recognition_QA, img_path)
        Anomaly_Recognition_answer = qwenvl_forward(args, Anomaly_Recognition_input, model, processor)

        # Image caption & model inference
        caption_instruction_input = construct_qwen_data(args, processor, caption_instruction, img_path)
        caption_instruction_answer = qwenvl_forward(args, caption_instruction_input, model, processor)
      
        answer_label_dict = ContraMetric(args, object_class_answer[0], object_class_QA, object_colour_answer[0], object_colour_QA, object_num_answer[0], object_num_QA, counterintuitive_answer[0], is_counterintuitive_QA, relation_answer[0], relation_QA, Anomaly_Recognition_answer[0], Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(dataset)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(dataset)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(dataset)}\n")
        logging.info(all_data)

    if pbar is not None:
        pbar.close()
