import torch
import time
from tqdm import tqdm
import csv
import logging
import os, sys
import copy
from torch.utils.data import Dataset
from typing import List, Dict, Tuple, Any
from argparse import Namespace
import json

import requests
from PIL import Image
from transformers import MllamaForConditionalGeneration, AutoProcessor
from huggingface_hub import login

from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary,
    extend_results as extend_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )

def load_model_and_processor(args: Namespace, model_path: str) -> tuple[MllamaForConditionalGeneration, AutoProcessor]:
    """
    Load a pre-trained Mllama model and its associated processor.

    Args:
        args (Namespace): Configuration arguments containing model parameter path.
        model_path (str): Path to the pre-trained model directory.

    Returns:
        tuple[MllamaForConditionalGeneration, AutoProcessor]:
            - Loaded Mllama model.
            - Corresponding processor for handling inputs.
    """
    # Authenticate Hugging Face account
    login(token='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')

    # Load the model with specified dtype and move it to GPU
    model = MllamaForConditionalGeneration.from_pretrained(
        model_path,
        torch_dtype=torch.bfloat16,
        cache_dir=args.model_para_path,
        ).cuda()
    
    # Load the processor for handling inputs
    processor = AutoProcessor.from_pretrained(model_path, cache_dir=args.model_para_path)

    return model, processor

class Llama32VisionInstructDataset(Dataset):
    """
    A dataset class for vision-language instruction tuning with LLaVA-NeXT.

    This dataset loads image-category-label pairs from a CSV file and processes them 
    into a format suitable for multimodal inference with vision-language models.
    """
    def __init__(self, args: Any, model: torch.nn.Module, processor: Any) -> None:
        """
        Initializes the dataset with given arguments, model, and processor.

        Args:
            args (Any): Arguments containing dataset configurations.
            model (torch.nn.Module): The multimodal model used for processing.
            processor (Any): The processor for text and image preprocessing.
        """
        self.img_category_pairs: List[Tuple[str, str, int]] = []  # Stores (img_id, category, label) tuples
        self.datasets_img_path: str = args.DATASET_PATH.get(args.dataset)
        self.model = model
        self.processor = processor
        self.args = args
 
        # Template message format for inference
        self.message = [
            {"role": "user", "content": [
                {"type": "image"},
                {"type": "text", "text": None}
            ]}
        ]

        self._load_img_label(args.DATASET_LABEL.get(args.dataset))
        self._get_image_filenames(args.DATASET_PATH.get(args.dataset))
    
    def _load_img_label(self, json_file_path):
        """Loads labels from the json file."""
        with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
            self.contrasense_json = json.load(jsonfile)
            
    def _get_image_filenames(self, folder_path, image_extensions=None):

        if image_extensions is None:
            image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

        self.image_filenames = []
        for filename in os.listdir(folder_path):
            if os.path.isfile(os.path.join(folder_path, filename)):
                ext = os.path.splitext(filename)[1].lower()
                if ext in image_extensions:
                    self.image_filenames.append(filename)
        
        logging.info(f"Found {len(self.image_filenames)} image(s) in '{folder_path}'.")

    def _get_questions(self, questions):
        question_keys = questions.keys()
        for question_key in question_keys:
            if "object class Question" in question_key:
                object_class_QA = questions[question_key]
            elif "object colour Question" in question_key:
                object_colour_QA = questions[question_key]
            elif "object num Question" in question_key:
                object_num_QA = questions[question_key]
            elif "is counterintuitive Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "relation Question" in question_key:
                relation_QA = questions[question_key]
            elif "Anomaly Recognition Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
            elif "function Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "Commonsense Reasoning Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
        return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA

    def __len__(self):
        return len(self.image_filenames)

    def __getitem__(self, idx: int) -> Dict[str, Any]:
        """
        Retrieves a single sample from the dataset and processes it.

        Args:
            idx (int): Index of the sample.

        Returns:
            Dict[str, Any]: A dictionary containing processed inputs and metadata.
        """

        img_id = self.image_filenames[idx]
        # Determine image path
        img_path = os.path.join(self.datasets_img_path, img_id)
        
        if not os.path.exists(img_path):
            raise FileNotFoundError(f"Image file not found: {img_path}")
        
        img_id = img_id.split('.')[0]
        
        data = self.contrasense_json.get(img_id)
        
        caption = data["caption"]
        questions = data["questions"]

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = self._get_questions(questions)
 
        return {
            'img_id'                        : img_id,
            'img_path'                      : img_path,
            'caption'                       : caption,
            'object_class_QA'               : object_class_QA,
            'object_colour_QA'              : object_colour_QA,
            'object_num_QA'                 : object_num_QA,
            'is_counterintuitive_QA'        : is_counterintuitive_QA,
            'relation_QA'                   : relation_QA,
            'Anomaly_Recognition_QA'        : Anomaly_Recognition_QA
        }

def llama32_vision_collate_fn(batch: List[Dict[str, Any]]) -> Dict[str, List[Any]]:
    """
    Custom collate function for batching vision data in LLama32 model.
    
    Args:
        batch (List[Dict[str, Any]]): A list of dictionaries, where each dictionary 
            contains the following keys:
            - 'inputs_presence': Presence-based input features
            - 'inputs_absence': Absence-based input features
            - 'img_id': Image identifiers
            - 'category': Category labels for each image
            - 'presence_label': Labels for presence-based questions
            - 'absence_label': Labels for absence-based questions
            - 'presence_question_template': Templates for presence-related questions
            - 'absence_question_template': Templates for absence-related questions

    Returns:
        Dict[str, List[Any]]: A dictionary containing batched data:
            - 'inputs_presence' (Any): Shared presence input features
            - 'inputs_absence' (Any): Shared absence input features
            - 'img_id' (List[Any]): List of image identifiers
            - 'category' (List[Any]): List of category labels
            - 'presence_label' (List[Any]): List of presence labels
            - 'absence_label' (List[Any]): List of absence labels
            - 'presence_question_template' (List[Any]): List of presence question templates
            - 'absence_question_template' (List[Any]): List of absence question templates
    """

    metadata = {
        'img_id'                        : [item['img_id'] for item in batch],
        'img_path'                      : [item['img_path'] for item in batch],
        'caption'                       : [item['caption'] for item in batch],
        'object_class_QA'               : [item['object_class_QA'] for item in batch],
        'object_colour_QA'              : [item['object_colour_QA'] for item in batch],
        'object_num_QA'                 : [item['object_num_QA'] for item in batch],
        'is_counterintuitive_QA'        : [item['is_counterintuitive_QA'] for item in batch],
        'relation_QA'                   : [item['relation_QA'] for item in batch],
        'Anomaly_Recognition_QA'        : [item['Anomaly_Recognition_QA'] for item in batch]
    }

    return {
        **metadata
    }

def _process_data(args, QA: str, img_path: str, processor, model) -> Dict[str, torch.Tensor]:
        """
        Prepares the input data for the model by processing the image and text.

        Args:
            question (str): The question prompt.
            img_path (str): Path to the image file.

        Returns:
            Dict[str, torch.Tensor]: The processed inputs suitable for the model.
        """
        if not os.path.exists(img_path):
            raise FileNotFoundError(f"Image file not found: {img_path}")

        try:
            if type(QA) == dict:
                question = "Please answer the following single-choice question by selecting the one correct option. Only output the letter of the correct answer, with no explanation." + "\n" + QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4] + "\n" + "Answer: "
            elif type(QA) == str:
                question = QA
        except:
            logging.info(QA)
            exit()

        message = [
            {"role": "user", "content": [
                {"type": "image"},
                {"type": "text", "text": None}
            ]}
        ]
        
        messages = copy.deepcopy(message)
        messages[0].update({"content": [
                {"type": "image"},
                {"type": "text", "text": question}
            ]})
        
        # Load image
        image = Image.open(img_path)
        image = image.resize((args.img_size, args.img_size), Image.BICUBIC)

        # Apply chat template to format input text
        input_text = processor.apply_chat_template(messages, add_generation_prompt=True)

        # Tokenize and preprocess image-text pair
        inputs = processor(
            image,
            input_text,
            add_special_tokens=False,
            return_tensors="pt"
        ).to(model.device)

        return inputs

def llama32_vision_forward(inputs: Dict[str, Any], model: Any, processor: Any) -> str:
    """
    Forward function for the LLama32 vision model to generate text responses.
    
    Args:
        inputs (Dict[str, Any]): Input dictionary containing preprocessed data for the model.
        model (Any): The LLama32 model instance, expected to support distributed execution.
        processor (Any): The processor instance used to decode the model's output.
    
    Returns:
        str: The final decoded text response after processing the model's output.
    """
    output = model.module.generate(**inputs, max_new_tokens=30)
    response = processor.decode(output[0]).split('\n')[-1].replace('<|eot_id|>', '')
    return response

def run_model(args: Any) -> None:
    """
    Runs the LLama3.2-11B-Vision-Instruct model for evaluation.
    
    Args:
        args (Any): Command-line arguments or configuration object containing required parameters.
    
    Raises:
        ValueError: If batch size is greater than 1, as it may degrade response quality.
    """
    model_path = args.module

    # Load model and processor
    model, processor = load_model_and_processor(args, model_path)

    # Validate batch size
    if args.batchsize > 1:
        logging.error(
            "Llama3.2-11B-Vision-Instruct might see a degradation in response quality with more images."
            )
        raise ValueError("Invalid batch size: Llama3.2-11B-Vision-Instruct requires batchsize to be 1.")

    torch.cuda.empty_cache()

    # Initialize dataset and dataloader
    dataset = Llama32VisionInstructDataset(args, model, processor)
    sampler = torch.utils.data.distributed.DistributedSampler(dataset, shuffle=False)
    data_loader = torch.utils.data.DataLoader(dataset, batch_size=args.batchsize, sampler=sampler, collate_fn=llama32_vision_collate_fn)

    # Enable Distributed Data Parallel (DDP)
    with torch.no_grad():
        model.cuda()
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.local_rank])
    
    start_time = time.time()

    # Initialize progress bar
    pbar = tqdm(total=len(data_loader), desc="Evaluating") if args.local_rank == 0 else None

    all_data = {}

    for idx, batch_data in enumerate(data_loader):
        if pbar is not None:
            pbar.update(1)
            pbar.set_postfix_str(f"GPU: {args.local_rank}")
        
        # Load batch data
        img_id                              = batch_data.get('img_id')[0]
        img_path                            = batch_data.get('img_path')[0]
        caption                             = batch_data.get('caption')
        object_class_QA                     = batch_data.get('object_class_QA')[0]
        object_colour_QA                    = batch_data.get('object_colour_QA')[0]
        object_num_QA                       = batch_data.get('object_num_QA')[0]
        is_counterintuitive_QA              = batch_data.get('is_counterintuitive_QA')[0]
        relation_QA                         = batch_data.get('relation_QA')[0]
        Anomaly_Recognition_QA              = batch_data.get('Anomaly_Recognition_QA')[0]
        caption_instruction                 = args.caption_instruction

        # Run inference
        # class question & model inference
        object_class_input = _process_data(args, object_class_QA, img_path, processor, model)
        object_class_answer = llama32_vision_forward(object_class_input, model, processor)

        # colour question & model inference
        object_colour_input = _process_data(args, object_colour_QA, img_path, processor, model)
        object_colour_answer = llama32_vision_forward(object_colour_input, model, processor)

        # num question & model inference
        object_num_input = _process_data(args, object_num_QA, img_path, processor, model)
        object_num_answer = llama32_vision_forward(object_num_input, model, processor)

        # counterintuitive question & model inference
        is_counterintuitive_input = _process_data(args, is_counterintuitive_QA, img_path, processor, model)
        counterintuitive_answer = llama32_vision_forward(is_counterintuitive_input, model, processor)

        # relation question & model inference
        relation_input = _process_data(args, relation_QA, img_path, processor, model)
        relation_answer = llama32_vision_forward(relation_input, model, processor)

        # anomaly recognition question & model inference
        Anomaly_Recognition_input = _process_data(args, Anomaly_Recognition_QA, img_path, processor, model)
        Anomaly_Recognition_answer = llama32_vision_forward(Anomaly_Recognition_input, model, processor)

        # Image caption & model inference
        caption_instruction_input = _process_data(args, caption_instruction, img_path, processor, model)
        caption_instruction_answer = llama32_vision_forward(caption_instruction_input, model, processor)

        # logging.info(Anomaly_Recognition_answer)
        

        answer_label_dict = ContraMetric(args, object_class_answer, object_class_QA, object_colour_answer, object_colour_QA, object_num_answer, object_num_QA, counterintuitive_answer, is_counterintuitive_QA, relation_answer, relation_QA, Anomaly_Recognition_answer, Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(dataset)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(dataset)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(dataset)}\n")
        logging.info(all_data)

    if pbar is not None:
        pbar.close()
