from PIL import Image
import requests
import copy
import csv
import torch
import os, sys
import logging
import time
from tqdm import tqdm
from torch.utils.data import Dataset
from typing import Any, Tuple, Dict, List
import json

from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary,
    extend_results as extend_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )

from transformers import PreTrainedTokenizer, PreTrainedModel

from llava.model.builder import load_pretrained_model
from llava.mm_utils import get_model_name_from_path, process_images, tokenizer_image_token
from llava.constants import IMAGE_TOKEN_INDEX, DEFAULT_IMAGE_TOKEN, DEFAULT_IM_START_TOKEN, DEFAULT_IM_END_TOKEN, IGNORE_INDEX
from llava.conversation import conv_templates, SeparatorStyle


def load_model_and_processor(
    args: object, model_path: str
) -> Tuple[PreTrainedTokenizer, PreTrainedModel, object, int]:
    """
    Load a pre-trained model, tokenizer, and image processor.

    Args:
        args (object): Argument object containing necessary configurations.
        model_path (str): Path to the pre-trained model directory.

    Returns:
        Tuple[PreTrainedTokenizer, PreTrainedModel, object, int]:
            - tokenizer: The tokenizer for text processing.
            - model: The pre-trained model instance.
            - image_processor: The image processor for handling images.
            - max_length: Maximum sequence length for tokenized inputs.
    """
    model_name = "llava_llama3"

    tokenizer, model, image_processor, max_length = load_pretrained_model(
        model_path, 
        None, 
        model_name, 
        device_map=None, 
        cache_dir=args.model_para_path) # Add any other thing you want to pass in llava_model_args
    return tokenizer, model, image_processor, max_length

class LLaVANeXTDataset(Dataset):
    """
    Custom dataset for LLaVA-NeXT, handling image-category-label pairs.
    """
    def __init__(
        self, args: object, tokenizer: object, model: object, image_processor: object
    ) -> None:
        """
        Initialize the dataset.

        Args:
            args (object): Configuration arguments.
            tokenizer (object): Tokenizer for text processing.
            model (object): Pre-trained model instance.
            image_processor (object): Image processor for handling images.
        """
        self.img_category_pairs = []    # Stores image-category pairs as (img_id, category, label)
        self.datasets_img_path = args.DATASET_PATH.get(args.dataset)  # Dataset path
        self.args = args
        self.tokenizer = tokenizer
        self.model = model
        self.image_processor = image_processor

        self._load_img_label(args.DATASET_LABEL.get(args.dataset))
        self._get_image_filenames(args.DATASET_PATH.get(args.dataset))
        
    def _prompt_process(self, QA) -> torch.Tensor:
        """
        Process prompt for the model.

        Args:
            question_template (str): Template for the prompt.

        Returns:
            torch.Tensor: Tokenized prompt tensor.
        """
        try:
            if type(QA) == dict:
                question = "Please answer the following single-choice question by selecting the one correct option. Only output the letter of the correct answer, with no explanation." + "\n" + QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4] + "\n" + "Answer: "
            elif type(QA) == str:
                question = QA
        except:
            logging.info(QA)
            exit()


        conv_template = "llava_llama_3"     # Ensure the correct chat template for different models
        question = DEFAULT_IMAGE_TOKEN + f"\n{question}"
        conv = copy.deepcopy(conv_templates[conv_template])
        conv.append_message(conv.roles[0], question)
        conv.append_message(conv.roles[1], None)
        conv.tokenizer = self.tokenizer
        prompt_question = conv.get_prompt()

        input_ids = tokenizer_image_token(prompt_question, self.tokenizer, IMAGE_TOKEN_INDEX, return_tensors="pt").unsqueeze(0).to(self.args.device)
        return input_ids


    def _load_img_label(self, json_file_path):
        """Loads labels from the json file."""
        with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
            self.contrasense_json = json.load(jsonfile)
            
    def _get_image_filenames(self, folder_path, image_extensions=None):
        if image_extensions is None:
            image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

        self.image_filenames = []
        for filename in os.listdir(folder_path):
            if os.path.isfile(os.path.join(folder_path, filename)):
                ext = os.path.splitext(filename)[1].lower()
                if ext in image_extensions:
                    self.image_filenames.append(filename)
        
        logging.info(f"Found {len(self.image_filenames)} image(s) in '{folder_path}'.")

    def _get_questions(self, questions):
        question_keys = questions.keys()
        for question_key in question_keys:
            if "object class Question" in question_key:
                object_class_QA = questions[question_key]
            elif "object colour Question" in question_key:
                object_colour_QA = questions[question_key]
            elif "object num Question" in question_key:
                object_num_QA = questions[question_key]
            elif "is counterintuitive Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "relation Question" in question_key:
                relation_QA = questions[question_key]
            elif "Anomaly Recognition Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
            elif "function Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "Commonsense Reasoning Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
        return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA

    def __len__(self):
        return len(self.image_filenames)

    def __getitem__(self, idx: int) -> Dict[str, Any]:
        """
        Retrieve a sample from the dataset.

        Args:
            idx (int): Index of the sample.

        Returns:
            Dict[str, Any]: Dictionary containing input tensors and metadata.
        """
        img_id = self.image_filenames[idx]
        # Determine image path
        img_path = os.path.join(self.datasets_img_path, img_id)
        
        if not os.path.exists(img_path):
            raise FileNotFoundError(f"Image file not found: {img_path}")
        
        img_id = img_id.split('.')[0]

        # logging.info(img_id)
        
        data = self.contrasense_json.get(img_id)
        
        caption = data["caption"]
        questions = data["questions"]

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = self._get_questions(questions)

        image = Image.open(img_path)
        image_tensor = process_images([image], self.image_processor, self.model.config)
        image_tensor = [_image.to(dtype=torch.float16, device=self.args.device) for _image in image_tensor]

        # Process texts
        object_class_input = self._prompt_process(object_class_QA)
        object_colour_input = self._prompt_process(object_colour_QA)
        object_num_input = self._prompt_process(object_num_QA)
        is_counterintuitive_input = self._prompt_process(is_counterintuitive_QA)
        relation_input = self._prompt_process(relation_QA)
        Anomaly_Recognition_input = self._prompt_process(Anomaly_Recognition_QA)
        caption_instruction_input = self._prompt_process(self.args.caption_instruction)
        
        image_sizes = [image.size]

        data = {
            'object_class_input'                    : object_class_input, 
            'object_colour_input'                   : object_colour_input, 
            'object_num_input'                      : object_num_input, 
            'is_counterintuitive_input'             : is_counterintuitive_input,
            'relation_input'                        : relation_input,
            'Anomaly_Recognition_input'             : Anomaly_Recognition_input, 
            'caption_instruction_input'             : caption_instruction_input, 
            'image_tensor'                          : image_tensor, 
            'img_id'                                : img_id, 
            'image_sizes'                           : image_sizes,
            'img_path'                              : img_path,
            'caption'                               : caption,
            'object_class_QA'                       : object_class_QA,
            'object_colour_QA'                      : object_colour_QA,
            'object_num_QA'                         : object_num_QA,
            'is_counterintuitive_QA'                : is_counterintuitive_QA,
            'relation_QA'                           : relation_QA,
            'Anomaly_Recognition_QA'                : Anomaly_Recognition_QA
            }
        return data

def LLaVANeXT_collate_fn(batch: List[Dict[str, Any]]) -> Dict[str, Any]:
    """
    Collate function for batching LLaVA-NeXT dataset samples.

    Args:
        batch (List[Dict[str, Any]]): List of samples from the dataset.

    Returns:
        Dict[str, Any]: Batched data with input tensors and metadata.
    """
    object_class_input = [item['object_class_input'] for item in batch][0]
    object_colour_input = [item['object_colour_input'] for item in batch][0]
    object_num_input = [item['object_num_input'] for item in batch][0]
    is_counterintuitive_input = [item['is_counterintuitive_input'] for item in batch][0]
    relation_input = [item['relation_input'] for item in batch][0]
    Anomaly_Recognition_input = [item['Anomaly_Recognition_input'] for item in batch][0]
    caption_instruction_input = [item['caption_instruction_input'] for item in batch][0]
    image_tensor = [item['image_tensor'] for item in batch][0]
    image_sizes = [item['image_sizes'] for item in batch]
    img_path = [item['img_path'] for item in batch]
    img_id = [item['img_id'] for item in batch]

    metadata = {
        'caption'                       : [item['caption'] for item in batch],
        'object_class_QA'               : [item['object_class_QA'] for item in batch],
        'object_colour_QA'              : [item['object_colour_QA'] for item in batch],
        'object_num_QA'                 : [item['object_num_QA'] for item in batch],
        'is_counterintuitive_QA'        : [item['is_counterintuitive_QA'] for item in batch],
        'relation_QA'                   : [item['relation_QA'] for item in batch],
        'Anomaly_Recognition_QA'        : [item['Anomaly_Recognition_QA'] for item in batch]
    }

    return {
            'object_class_input'                    : object_class_input, 
            'object_colour_input'                   : object_colour_input, 
            'object_num_input'                      : object_num_input, 
            'is_counterintuitive_input'             : is_counterintuitive_input,
            'relation_input'                        : relation_input,
            'Anomaly_Recognition_input'             : Anomaly_Recognition_input, 
            'caption_instruction_input'             : caption_instruction_input, 
            'image_tensor'                          : image_tensor, 
            'img_id'                                : img_id, 
            'image_sizes'                           : image_sizes,
            'img_path'                              : img_path,
            **metadata
            }

def llava_next_forward(
    input_ids: torch.Tensor,
    image_tensor: torch.Tensor,
    image_sizes: List[Any],
    model: PreTrainedModel,
    tokenizer: PreTrainedTokenizer
) -> List[str]:
    """
    Perform forward inference with the LLaVA-Next model.

    Args:
        input_ids (torch.Tensor): Tokenized input text tensor.
        image_tensor (torch.Tensor): Preprocessed image tensor.
        image_sizes (List[Any]): List of image sizes.
        model (PreTrainedModel): The LLaVA-Next model.
        tokenizer (PreTrainedTokenizer): Tokenizer for decoding output.

    Returns:
        List[str]: Decoded text outputs from the model.
    """
    cont = model.module.generate(
        input_ids,
        images=image_tensor,
        image_sizes=image_sizes,
        do_sample=False,
        temperature=0,
        max_new_tokens=256,
        )
    
    text_outputs = tokenizer.batch_decode(cont, skip_special_tokens=True)
    return text_outputs

def run_model(args: Any) -> None:
    """
    Run the LLaVA-NeXT model for inference and evaluation in a distributed setting.

    Args:
        args (Any): Configuration object containing dataset, model, and training parameters.

    Raises:
        ValueError: If batch size is greater than 1, since LLaVA-NeXT does not support batch processing.
    """
    model_path = args.module

    # Load model and processor
    tokenizer, model, image_processor, _ = load_model_and_processor(args, model_path)

    # Validate batch size
    if args.batchsize > 1:
        logging.error(
            "LLaVA-NeXT does not support batched processing of independent image-text pairs. Please set batchsize to 1."
            )
        raise ValueError("Invalid batch size: LLaVA-NeXT requires batchsize to be 1.")

    # Initialize dataset and dataloader
    dataset = LLaVANeXTDataset(args, tokenizer, model, image_processor)
    sampler = torch.utils.data.distributed.DistributedSampler(dataset, shuffle=False)
    data_loader = torch.utils.data.DataLoader(
        dataset, 
        batch_size=args.batchsize, 
        sampler=sampler, 
        collate_fn=LLaVANeXT_collate_fn
        )
    
    # Enable Distributed Data Parallel (DDP)
    with torch.no_grad():
        model.cuda()
        model = torch.nn.parallel.DistributedDataParallel(model, device_ids=[args.local_rank])
    
    start_time = time.time()

    # Initialize progress bar
    pbar = tqdm(total=len(data_loader), desc="Evaluating") if args.local_rank == 0 else None

    all_data = {}

    for idx, batch_data in enumerate(data_loader):
        if pbar is not None:
            pbar.update(1)
            pbar.set_postfix_str(f"GPU: {args.local_rank}")

        # Load batch data
        object_class_input                  = batch_data.get('object_class_input')
        object_colour_input                 = batch_data.get('object_colour_input')
        object_num_input                    = batch_data.get('object_num_input')
        is_counterintuitive_input           = batch_data.get('is_counterintuitive_input')
        relation_input                      = batch_data.get('relation_input')
        Anomaly_Recognition_input           = batch_data.get('Anomaly_Recognition_input')
        caption_instruction_input           = batch_data.get('caption_instruction_input')
        image_tensor                        = batch_data.get('image_tensor')[0].to(args.device)
        image_sizes                         = batch_data.get('image_sizes')
        img_id                              = batch_data.get('img_id')[0]
        img_path                            = batch_data.get('img_path')
        caption                             = batch_data.get('caption')
        object_class_QA                     = batch_data.get('object_class_QA')[0]
        object_colour_QA                    = batch_data.get('object_colour_QA')[0]
        object_num_QA                       = batch_data.get('object_num_QA')[0]
        is_counterintuitive_QA              = batch_data.get('is_counterintuitive_QA')[0]
        relation_QA                         = batch_data.get('relation_QA')[0]
        Anomaly_Recognition_QA              = batch_data.get('Anomaly_Recognition_QA')[0]

        # Model inference
        object_class_answer = llava_next_forward(object_class_input, image_tensor, image_sizes, model, tokenizer)
        object_colour_answer = llava_next_forward(object_colour_input, image_tensor, image_sizes, model, tokenizer)
        object_num_answer = llava_next_forward(object_num_input, image_tensor, image_sizes, model, tokenizer)
        counterintuitive_answer = llava_next_forward(is_counterintuitive_input, image_tensor, image_sizes, model, tokenizer)
        relation_answer = llava_next_forward(relation_input, image_tensor, image_sizes, model, tokenizer)
        Anomaly_Recognition_answer = llava_next_forward(Anomaly_Recognition_input, image_tensor, image_sizes, model, tokenizer)
        caption_instruction_answer = llava_next_forward(caption_instruction_input, image_tensor, image_sizes, model, tokenizer)

        answer_label_dict = ContraMetric(args, object_class_answer[0], object_class_QA, object_colour_answer[0], object_colour_QA, object_num_answer[0], object_num_QA, counterintuitive_answer[0], is_counterintuitive_QA, relation_answer[0], relation_QA, Anomaly_Recognition_answer[0], Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(dataset)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(dataset)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(dataset)}\n")
        logging.info(all_data)

    if pbar is not None:
        pbar.close()
