import time
from tqdm import tqdm
import csv
import logging
import os, sys
import torch
from typing import Tuple, List, Any
import json
from PIL import Image

from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary, 
    extend_results as extend_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )

import PIL.Image
import google.generativeai as genai


def process_img_data(args: Any, img_id: str) -> Tuple[PIL.Image.Image, str, str]:
    """
    Processes image data and generates prompts based on the given category.
    
    Args:
        args (Any): The argument object containing dataset configurations.
        img_id (str): The identifier of the image file.
        category (str): The category name to be inserted into the prompts.
    
    Returns:
        Tuple[PIL.Image.Image, str, str]:
            - The loaded image as a PIL Image object.
            - The presence prompt with the category included.
            - The absence prompt with the category included.
    
    Example:
        >>> args = Namespace(DATASET_PATH={"dataset": "/path/to/dataset"},
        ...                 dataset="dataset",
        ...                 presence_question_template="Is there a [class] in the image?",
        ...                 absence_question_template="Is there no [class] in the image?")
        >>> img_id = "image1.jpg"
        >>> category = "cat"
        >>> image, prompt_presence, prompt_absence = process_data(args, img_id, category)
    """
    # Determine image path
    img_path = os.path.join(args.DATASET_PATH.get(args.dataset), img_id)

    # Load image
    image = PIL.Image.open(img_path)

    if args.img_size is not None:
        image = image.resize((args.img_size, args.img_size), Image.BICUBIC)
    
    return image

def process_QA_data(QA):
    try:
        if type(QA) == dict:
            question = "Please answer the following single-choice question by selecting the one correct option. Only output the letter of the correct answer, with no explanation." + "\n" + QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4] + "\n" + "Answer: "
        elif type(QA) == str:
            question = QA
    except:
        logging.info(QA)
        exit()

    return question

def gemini_1_5_pro_forward(model: Any, image: Any, prompt: str) -> Any:
    """
    Sends an image and a text prompt to the Gemini 1.5 Pro model for content generation.
    
    Args:
        model (Any): The Gemini 1.5 Pro model instance.
        image (Any): The image input, format depends on model requirements.
        prompt (str): The text prompt to guide the model's response.
    
    Returns:
        Any: The response generated by the model.
    
    Example:
        >>> model = GeminiModel()
        >>> image = load_image("example.jpg")
        >>> prompt = "Describe the objects in the image."
        >>> response = gemini_1_5_pro_forward(model, image, prompt)
    """
    response = model.generate_content([prompt, image])
    return response

def gemini_2_0_flash_forward(model: Any, image: Any, prompt: str) -> Any:
    """
    Sends an image and a text prompt to the Gemini 2.0 Flash model for content generation.
    
    Args:
        model (Any): The Gemini 2.0 Flash model instance.
        image (Any): The image input, format depends on model requirements.
        prompt (str): The text prompt to guide the model's response.
    
    Returns:
        Any: The response generated by the model.
    
    Example:
        >>> model = GeminiFlashModel()
        >>> image = load_image("example.jpg")
        >>> prompt = "Describe the objects in the image."
        >>> response = gemini_2_0_flash_forward(model, image, prompt)
    """
    response = model.generate_content(
        contents=[prompt, image])
    return response

def _load_img_label(json_file_path):
    """Loads labels from the json file."""
    with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
        contrasense_json = json.load(jsonfile)
    return contrasense_json
        
            
def _get_image_filenames(folder_path, image_extensions=None):
    if image_extensions is None:
        image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

    image_filenames = []
    for filename in os.listdir(folder_path):
        if os.path.isfile(os.path.join(folder_path, filename)):
            ext = os.path.splitext(filename)[1].lower()
            if ext in image_extensions:
                image_filenames.append(filename)
    
    logging.info(f"Found {len(image_filenames)} image(s) in '{folder_path}'.")
    return image_filenames

def _get_questions(questions):
    question_keys = questions.keys()
    for question_key in question_keys:
        if "object class Question" in question_key:
            object_class_QA = questions[question_key]
        elif "object colour Question" in question_key:
            object_colour_QA = questions[question_key]
        elif "object num Question" in question_key:
            object_num_QA = questions[question_key]
        elif "is counterintuitive Question" in question_key:
            is_counterintuitive_QA = questions[question_key]
        elif "relation Question" in question_key:
            relation_QA = questions[question_key]
        elif "Anomaly Recognition Question" in question_key:
            Anomaly_Recognition_QA = questions[question_key]
        elif "function Question" in question_key:
            is_counterintuitive_QA = questions[question_key]
        elif "Commonsense Reasoning Question" in question_key:
            Anomaly_Recognition_QA = questions[question_key]
    return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA

def run_model(args: Any) -> None:
    """
    Runs the generative model on a dataset and evaluates its performance.
    
    Args:
        args (Any): The argument object containing dataset configurations and model parameters.
    
    Returns:
        None: The function does not return a value, but logs performance metrics and saves prediction tensors.
    
    Example:
        >>> args = Namespace(module="gemini-1.5-pro", dataset_csv_path="/path/to/dataset.csv")
        >>> run_model(args)
    """
    model_path = args.module

    # Load model
    genai.configure(api_key="Your_API_Key")
    model = genai.GenerativeModel(model_name=model_path)

    # Prepare data
    contrasense_json = _load_img_label(args.DATASET_LABEL.get(args.dataset))
    image_filenames = _get_image_filenames(args.DATASET_PATH.get(args.dataset))

    start_time = time.time()

    # Storage for predictions and labels
    all_data = {}

    for idx, img_name in enumerate(tqdm(image_filenames)):
        # Load batch data
        img_id = img_name.split('.')[0]
        data = contrasense_json.get(img_id)
        
        caption = data["caption"]
        questions = data["questions"]
        caption_instruction = args.caption_instruction

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = _get_questions(questions)

        # Process data for model
        image = process_img_data(args, img_name)

        object_class_message = process_QA_data(object_class_QA)
        object_colour_message = process_QA_data(object_colour_QA)
        object_num_message = process_QA_data(object_num_QA)
        is_counterintuitive_message = process_QA_data(is_counterintuitive_QA)
        relation_message = process_QA_data(relation_QA)
        Anomaly_Recognition_message = process_QA_data(Anomaly_Recognition_QA)
        caption_instruction_message = process_QA_data(caption_instruction)
        

        try:
            if model_path == "gemini-1.5-pro":
                object_class_answer = gemini_1_5_pro_forward(model, image, object_class_message)
                object_colour_answer = gemini_1_5_pro_forward(model, image, object_colour_message)
                object_num_answer = gemini_1_5_pro_forward(model, image, object_num_message)
                is_counterintuitive_answer = gemini_1_5_pro_forward(model, image, is_counterintuitive_message)
                relation_answer = gemini_1_5_pro_forward(model, image, relation_message)
                Anomaly_Recognition_answer = gemini_1_5_pro_forward(model, image, Anomaly_Recognition_message)
                caption_instruction_answer = gemini_1_5_pro_forward(model, image, caption_instruction_message)

            elif model_path == "gemini-2.0-flash":
                object_class_answer = gemini_2_0_flash_forward(model, image, object_class_message)
                object_colour_answer = gemini_2_0_flash_forward(model, image, object_colour_message)
                object_num_answer = gemini_2_0_flash_forward(model, image, object_num_message)
                is_counterintuitive_answer = gemini_2_0_flash_forward(model, image, is_counterintuitive_message)
                relation_answer = gemini_2_0_flash_forward(model, image, relation_message)
                Anomaly_Recognition_answer = gemini_2_0_flash_forward(model, image, Anomaly_Recognition_message)
                caption_instruction_answer = gemini_2_0_flash_forward(model, image, caption_instruction_message)

            if isinstance(object_class_answer.text, str) and isinstance(object_colour_answer.text, str) and isinstance(object_num_answer.text, str) and isinstance(is_counterintuitive_answer.text, str) and isinstance(relation_answer.text, str) and isinstance(Anomaly_Recognition_answer.text, str) and isinstance(caption_instruction_answer.text, str):
                object_class_answer = object_class_answer.text
                object_colour_answer = object_colour_answer.text
                object_num_answer = object_num_answer.text
                counterintuitive_answer = is_counterintuitive_answer.text
                relation_answer = relation_answer.text
                Anomaly_Recognition_answer = Anomaly_Recognition_answer.text
                caption_instruction_answer = caption_instruction_answer.text
            else:
                logging.info(f'Error: Response is empty or malformed for data {data}')
                raise ValueError("Response is empty or malformed")

        except (ValueError, KeyError) as e:
            logging.info(f'Error: Response is empty or malformed for data {data}')
            continue

        answer_label_dict = ContraMetric(args, object_class_answer, object_class_QA, object_colour_answer, object_colour_QA, object_num_answer, object_num_QA, counterintuitive_answer, is_counterintuitive_QA, relation_answer, relation_QA, Anomaly_Recognition_answer, Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(all_data)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(all_data)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(all_data)}\n")
        logging.info(all_data)
