import time
from tqdm import tqdm
import csv
import logging
import os, sys
import base64
import torch
from typing import Tuple, List, Dict, Any
import json
from PIL import Image
import io

from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary, 
    extend_results as extend_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    encode_image as encode_image,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )

from openai import OpenAI
import openai

model_key_api = "Your_API_Key"

def _load_img_label(json_file_path):
    """Loads labels from the json file."""
    with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
        contrasense_json = json.load(jsonfile)
    return contrasense_json
        
            
def _get_image_filenames(folder_path, image_extensions=None):
    if image_extensions is None:
        image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

    image_filenames = []
    for filename in os.listdir(folder_path):
        if os.path.isfile(os.path.join(folder_path, filename)):
            ext = os.path.splitext(filename)[1].lower()
            if ext in image_extensions:
                image_filenames.append(filename)
    
    logging.info(f"Found {len(image_filenames)} image(s) in '{folder_path}'.")
    return image_filenames

def _get_questions(questions):
    question_keys = questions.keys()
    for question_key in question_keys:
        if "object class Question" in question_key:
            object_class_QA = questions[question_key]
        elif "object colour Question" in question_key:
            object_colour_QA = questions[question_key]
        elif "object num Question" in question_key:
            object_num_QA = questions[question_key]
        elif "is counterintuitive Question" in question_key:
            is_counterintuitive_QA = questions[question_key]
        elif "relation Question" in question_key:
            relation_QA = questions[question_key]
        elif "Anomaly Recognition Question" in question_key:
            Anomaly_Recognition_QA = questions[question_key]
        elif "function Question" in question_key:
            is_counterintuitive_QA = questions[question_key]
        elif "Commonsense Reasoning Question" in question_key:
            Anomaly_Recognition_QA = questions[question_key]
    return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA


def formalized_message(base64_image: str, question: str) -> List[Dict[str, Any]]:
    """
    Constructs a structured message payload containing a text question and an image in base64 format.
    
    Args:
        base64_image (str): The base64-encoded string of the image.
        question (str): The text-based question related to the image.
    
    Returns:
        List[Dict[str, Any]]: A structured message payload following a predefined format.
    
    Example:
        >>> message = formalized_message("/9j/4AAQSkZJRg...", "What is in this image?")
        >>> print(message)
        [{
            "role": "user",
            "content": [
                {"type": "text", "text": "What is in this image?"},
                {"type": "image_url", "image_url": {"url": "data:image/jpeg;base64,/9j/4AAQSkZJRg..."}}
            ]
        }]
    """
    return [
        {
            "role": "user",
            "content": [
                {"type": "text", "text": question},
                {"type": "image_url", "image_url": {"url": f"data:image/jpeg;base64,{base64_image}"}}
            ]
        }
    ]

def resize_and_encode_image(img_path: str, resize_size: Tuple[int, int]) -> str:
    """
    Resize the input image to a fixed size and return base64 encoded string.

    Args:
        img_path (str): Path to the image file.
        resize_size (Tuple[int, int]): Target (width, height).

    Returns:
        str: Base64-encoded PNG image string.
    """
    with Image.open(img_path) as img:
        img = img.resize(resize_size, Image.BICUBIC)
        buffer = io.BytesIO()
        img.save(buffer, format="PNG")
        return base64.b64encode(buffer.getvalue()).decode("utf-8")

def prepare_data(args: Any, img_id: str, QA: str) -> Tuple[Dict[str, Any], Dict[str, Any]]:
    """
    Prepares data for image-based question answering by constructing structured messages.
    
    Args:
        args (Any): The arguments object containing dataset path and prompt templates.
        img_id (str): The unique identifier of the image.
        category (str): The category name to be used in the prompts.
    
    Returns:
        Tuple[Dict[str, Any], Dict[str, Any]]: A tuple containing two structured messages:
            - message_presence: A message asking about the presence of the category in the image.
            - message_absence: A message asking about the absence of the category in the image.
    
    Example:
        >>> args.dataset = "sample_dataset"
        >>> args.DATASET_PATH = {"sample_dataset": "/path/to/dataset"}
        >>> args.presence_question_template = "Is there a [class] in the image?"
        >>> args.absence_question_template = "Is there no [class] in the image?"
        >>> prepare_data(args, "image_001.jpg", "cat")
        ({"role": "user", "content": [...]}, {"role": "user", "content": [...]})
    """

    try:
        if type(QA) == dict:
            question = "Please answer the following single-choice question by selecting the one correct option. Only output the letter of the correct answer, with no explanation." + "\n" + QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4] + "\n" + "Answer: "
        elif type(QA) == str:
            question = QA
    except:
        logging.info(QA)
        exit()

    # Determine image path
    img_path = os.path.join(args.DATASET_PATH.get(args.dataset), img_id)

    # Encode image to base64
    # base64_image = encode_image(img_path)
    base64_image = resize_and_encode_image(img_path, resize_size=(args.img_size, args.img_size))

    # Construct messages
    message = formalized_message(base64_image, question)


    return message

def gpt_series_forward(client: Any, model_path: str, message: List[Dict[str, Any]]) -> str:
    """
    Sends a message to the GPT-4o model and retrieves the response.
    
    Args:
        client (Any): The API client used to communicate with the GPT model.
        model_path (str): The identifier or path of the GPT model to be used.
        message (List[Dict[str, Any]]): A structured message list in the required format.
    
    Returns:
        str: The text content of the response from the GPT model.
    
    Example:
        >>> client = SomeAPIClient()
        >>> model_path = "gpt-4o"
        >>> message = [{"role": "user", "content": "Hello, GPT!"}]
        >>> response = gpt_4o_forward(client, model_path, message)
        >>> print(response)
        "Hello! How can I assist you today?"
    """
    response = client.chat.completions.create(
        model=model_path,
        messages=message,
        timeout=60
        )
    response_content = response.choices[0].message.content
    return response_content

def run_model(args: Any) -> None:
    """
    Runs the GPT series model for image classification evaluation using a dataset.
    
    Args:
        args (Any): The argument object containing necessary configurations such as dataset path,
                    model details, and processing parameters.
    
    Returns:
        None: This function processes and evaluates predictions without returning values.
    
    Example:
        >>> args = Namespace(module="gpt-4o", dataset_csv_path="data.csv", nproc_per_node=4)
        >>> run_model(args)
    """
    model_path = args.module
    client = OpenAI(api_key=model_key_api)

    response = client.chat.completions.create(
        model="gpt-4o",
        messages=[
            {"role": "user", "content": "Hello!"}
        ]
    )

    # logging model version
    logging.info(f"-Model Version: {response.model}")


    # Prepare data
    contrasense_json = _load_img_label(args.DATASET_LABEL.get(args.dataset))
    image_filenames = _get_image_filenames(args.DATASET_PATH.get(args.dataset))

    start_time = time.time()

    # Storage for predictions and labels
    all_data = {}

    for idx, img_name in enumerate(tqdm(image_filenames)):
        # Load batch data

        img_id = img_name.split('.')[0]
        data = contrasense_json.get(img_id)
        
        caption = data["caption"]
        questions = data["questions"]
        caption_instruction = args.caption_instruction

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = _get_questions(questions)

        # Prepare messages for model
        object_class_message = prepare_data(args, img_name, object_class_QA)
        object_colour_message = prepare_data(args, img_name, object_colour_QA)
        object_num_message = prepare_data(args, img_name, object_num_QA)
        is_counterintuitive_message = prepare_data(args, img_name, is_counterintuitive_QA)
        relation_message = prepare_data(args, img_name, relation_QA)
        Anomaly_Recognition_message = prepare_data(args, img_name, Anomaly_Recognition_QA)
        caption_instruction_message = prepare_data(args, img_name, caption_instruction)

        try:
            object_class_answer = gpt_series_forward(client, model_path, object_class_message)
            object_colour_answer = gpt_series_forward(client, model_path, object_colour_message)
            object_num_answer = gpt_series_forward(client, model_path, object_num_message)
            counterintuitive_answer = gpt_series_forward(client, model_path, is_counterintuitive_message)
            relation_answer = gpt_series_forward(client, model_path, relation_message)
            Anomaly_Recognition_answer = gpt_series_forward(client, model_path, Anomaly_Recognition_message)
            caption_instruction_answer = gpt_series_forward(client, model_path, caption_instruction_message)

        except Exception as e:
            logging.info(f'Error during request: {e} for image pair {img_id}')
            continue

        try:
            _, _, _, _, _, _ = object_class_answer[0], object_colour_answer[0], object_num_answer[0], counterintuitive_answer[0], relation_answer[0], Anomaly_Recognition_answer[0]
        except Exception as e:
            logging.info(f"Error during request: {e} for image {img_id}.")
            continue
        
        answer_label_dict = ContraMetric(args, object_class_answer, object_class_QA, object_colour_answer, object_colour_QA, object_num_answer, object_num_QA, counterintuitive_answer, is_counterintuitive_QA, relation_answer, relation_QA, Anomaly_Recognition_answer, Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(all_data)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(all_data)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(all_data)}\n")
        logging.info(all_data)
