import os, sys
import torch
import time
from tqdm import tqdm
import json
import logging
import os
os.environ["TOKENIZERS_PARALLELISM"] = "false"

import copy
import random
from torch.utils.data import Dataset
import torch.distributed as dist
from scripts.utils import (
    convert_responses_to_binary as convert_responses_to_binary, 
    extend_results as extend_results,
    metric_performances as metric_performances,
    format_elapsed_time as format_elapsed_time,
    construct_tensor_path as construct_tensor_path,
    store_tensor as store_tensor,
    match_option_from_caption as match_option_from_caption,
    ContraMetric as ContraMetric
    )
import torch.nn as nn
from typing import Any, Tuple
import deepspeed

from transformers import AutoModelForCausalLM


base_conversation = [
        {"role": "User", "content": "Compare and contrast <image_placeholder>", "images": ["path/to/image"]},
        {"role": "Assistant", "content": ""}
    ]

def deepseek_vl_collate_fn(batch):
    """
    Collate function for DeepSeek-VL dataset batches.
    
    Processes and batches different types of data from individual samples:
    - Stacks visual inputs into tensors
    - Preserves list structure for metadata
    - Handles both presence and absence modality data
    
    Args:
        batch: List of dataset items containing dictionary entries
    
    Returns:
        Dictionary containing batched tensors and metadata lists
    """

    
    # Collect metadata while preserving list structure
    metadata = {
        'img_id'                        : [item['img_id'] for item in batch],
        'img_path'                      : [item['img_path'] for item in batch],
        'caption'                       : [item['caption'] for item in batch],
        'object_class_QA'               : [item['object_class_QA'] for item in batch],
        'object_colour_QA'              : [item['object_colour_QA'] for item in batch],
        'object_num_QA'                 : [item['object_num_QA'] for item in batch],
        'is_counterintuitive_QA'        : [item['is_counterintuitive_QA'] for item in batch],
        'relation_QA'                   : [item['relation_QA'] for item in batch],
        'Anomaly_Recognition_QA'        : [item['Anomaly_Recognition_QA'] for item in batch]
    }
    
    return {
        **metadata
    }

class DeepSeekVL7BChatDataset(Dataset):
    """
    Dataset for DeepSeek-VL-7B-Chat model evaluation with OOD detection.
    
    Args:
        csv_file_path (str): Path to the CSV file containing image-category-label pairs.
        datasets_img_path (str): Path to the dataset images.
        vl_chat_processor (object): Processor for handling vision-language inputs.
        vl_gpt (object): Vision-language model instance.
    """
    def __init__(self, args, vl_chat_processor, vl_gpt):
        if not os.path.exists(args.DATASET_LABEL.get(args.dataset)):
            raise FileNotFoundError(f"LABEL file not found: {args.DATASET_LABEL}")
        if not os.path.exists(args.DATASET_PATH.get(args.dataset)):
            raise FileNotFoundError(f"Dataset image path not found: {args.DATASET_PATH.get(args.DATASET_LABEL)}")
        self.args = args
        self.img_category_pairs = []  # Stores image-category-label tuples
        self.datasets_img_path = args.DATASET_PATH.get(args.dataset)
        self.processor = vl_chat_processor
        self.vl_gpt = vl_gpt
        
        # Load image-category-label pairs
        self._load_img_label(args.DATASET_LABEL.get(args.dataset))
        self._get_image_filenames(args.DATASET_PATH.get(args.dataset))
    
    def _load_img_label(self, json_file_path):
        """Loads labels from the json file."""
        with open(json_file_path, 'r', newline='', encoding='utf-8') as jsonfile:
            self.contrasense_json = json.load(jsonfile)
            
    def _get_image_filenames(self, folder_path, image_extensions=None):
        if image_extensions is None:
            image_extensions = {'.jpg', '.jpeg', '.png', '.bmp', '.gif', '.tiff', '.webp'}

        self.image_filenames = []
        for filename in os.listdir(folder_path):
            if os.path.isfile(os.path.join(folder_path, filename)):
                ext = os.path.splitext(filename)[1].lower()
                if ext in image_extensions:
                    self.image_filenames.append(filename)
        
        logging.info(f"Found {len(self.image_filenames)} image(s) in '{folder_path}'.")

    def _get_questions(self, questions):
        question_keys = questions.keys()

        for question_key in question_keys:
            if "object class Question" in question_key:
                object_class_QA = questions[question_key]
            elif "object colour Question" in question_key:
                object_colour_QA = questions[question_key]
            elif "object num Question" in question_key:
                object_num_QA = questions[question_key]
            elif "is counterintuitive Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "function Question" in question_key:
                is_counterintuitive_QA = questions[question_key]
            elif "relation Question" in question_key:
                relation_QA = questions[question_key]
            elif "Anomaly Recognition Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
            elif "Commonsense Reasoning Question" in question_key:
                Anomaly_Recognition_QA = questions[question_key]
        return object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA

    def __len__(self):
        return len(self.image_filenames)
    
    def __getitem__(self, idx):
        img_id = self.image_filenames[idx]
        # Determine image path
        img_path = os.path.join(self.datasets_img_path, img_id)
        
        if not os.path.exists(img_path):
            raise FileNotFoundError(f"Image file not found: {img_path}")
        
        img_id = img_id.split('.')[0]

        # logging.info(img_id)
        
        data = self.contrasense_json.get(img_id)

        # logging.info(data)
        
        caption = data["caption"]
        questions = data["questions"]

        object_class_QA, object_colour_QA, object_num_QA, is_counterintuitive_QA, relation_QA, Anomaly_Recognition_QA = self._get_questions(questions)
 
        return {
            'img_id'                        : img_id,
            'img_path'                      : img_path,
            'caption'                       : caption,
            'object_class_QA'               : object_class_QA,
            'object_colour_QA'              : object_colour_QA,
            'object_num_QA'                 : object_num_QA,
            'is_counterintuitive_QA'        : is_counterintuitive_QA,
            'relation_QA'                   : relation_QA,
            'Anomaly_Recognition_QA'        : Anomaly_Recognition_QA
        }

def deepseekvl_format_data(args, processor, vl_gpt, QA, img_path):
    if "VL2" in args.eval_model:
        from deepseek_vl2.utils.io import load_pil_images
        load_pil_images = load_pil_images
    else:
        from deepseek_vl.utils.io import load_pil_images
        load_pil_images = load_pil_images
    

    try:
        if type(QA) == dict:
            question = QA["question"] + "\n" + QA["options"][0] + "\n" + QA["options"][1] + "\n" + QA["options"][2] + "\n" + QA["options"][3] + "\n" + QA["options"][4]
        elif type(QA) == str:
            question = QA
    except:
        logging.info(QA)
        exit()


    base_conversation = [
            {"role": "User", "content": "Compare and contrast <image_placeholder>", "images": ["path/to/image"]},
            {"role": "Assistant", "content": ""}
        ] if "VL2" not in args.eval_model else [
            {"role": "<|User|>", "content": "Text <image>", "images": ["path/to/image"]},
            {"role": "<|Assistant|>", "content": ""},
        ]
    base_conversation[0].update(
            {"content": f"<image_placeholder>{question}", "images": [img_path]} 
            if "VL2" not in args.eval_model else 
            {"content": f"{question}<image>", "images": [img_path]}
            )
        
    # Process images
    pil_images = load_pil_images(base_conversation)
    
    inputs = processor(conversations=base_conversation, images=pil_images, force_batchify=True).to(vl_gpt.device)
    return inputs
        

def deepseekvl_forward(
    args,
    vl_gpt, 
    tokenizer: Any, 
    inputs
) -> torch.Tensor:
    """
    Forward function for DeepSeek-VL model to generate responses from image-text inputs.
    
    Args:
        vl_gpt (MultiModalityCausalLM): The multimodal language model.
        tokenizer (Any): The tokenizer associated with the model.
        inputs (BatchedVLChatProcessorOutput): Processed inputs including image and text features.
    
    Returns:
        torch.Tensor: The generated token IDs representing the response.
    """
    try:
        # Ensure the model is in evaluation mode
        # vl_gpt.eval()

        # Run image encoder to get the image embeddings
        inputs_embeds = vl_gpt.module.prepare_inputs_embeds(**inputs) \
            if args.ddp and "VL2" not in args.eval_model else \
            vl_gpt.prepare_inputs_embeds(**inputs)

        """
            Warning!!! DeepSeek-VL2 version and version 1.0 have modifications to the vl_gpt module, 
            vl_gpt.language_model.generate --> vl_gpt.language.generate
        """

        # Run the language model to generate a response
        outputs = vl_gpt.module.language_model.generate(
            inputs_embeds=inputs_embeds,
            attention_mask=inputs.attention_mask,
            pad_token_id=tokenizer.eos_token_id,
            bos_token_id=tokenizer.bos_token_id,
            eos_token_id=tokenizer.eos_token_id,
            max_new_tokens=512,
            do_sample=False,
            use_cache=True
        ) if args.ddp and "VL2" not in args.eval_model else \
        vl_gpt.language.generate(
            inputs_embeds=inputs_embeds,
            attention_mask=inputs.attention_mask,
            pad_token_id=tokenizer.eos_token_id,
            bos_token_id=tokenizer.bos_token_id,
            eos_token_id=tokenizer.eos_token_id,
            max_new_tokens=512,
            do_sample=False,
            use_cache=True
        )
        # Decode the generated output into text
        answer = tokenizer.decode(outputs[0].cpu().tolist(), skip_special_tokens=True)
        return answer
    
    except Exception as e:
        logging.error(f"Error in deepseekvl_forward: {e}")
        return None

def load_model_and_processor(args, model_path: str) -> Tuple[object, object, object]:
    """
    Loads the appropriate model and processor based on evaluation model type.
    
    Args:
        args: Configuration arguments containing model specifications
        model_path: Path to the pretrained model directory
        
    Returns:
        Tuple containing:
        - Processor for visual language chat tasks
        - Tokenizer for text processing
        - Loaded multi-modality language model
        
    Raises:
        ValueError: If CUDA is unavailable or model loading fails
    """
    # Check CUDA availability
    if not torch.cuda.is_available():
        raise ValueError("CUDA is required but not available")

    # Load model components based on specified model type
    if args.eval_model == "DeepSeek-VL-7B-Chat":
        # Load components for DeepSeek-VL-7B-Chat variant
        from deepseek_vl.models.processing_vlm import BatchedVLChatProcessorOutput
        from deepseek_vl.models import VLChatProcessor, MultiModalityCausalLM
        # from deepseek_vl.utils.io import load_pil_images

        vl_chat_processor = VLChatProcessor.from_pretrained(
            model_path, 
            cache_dir=args.model_para_path
        )
        vl_gpt = AutoModelForCausalLM.from_pretrained(
            model_path,
            trust_remote_code=True,
            cache_dir=args.model_para_path
        )
    else:
        # Load components for DeepSeek-VLv2 variant
        from deepseek_vl2.models import DeepseekVLV2Processor, DeepseekVLV2ForCausalLM
        # from deepseek_vl2.utils.io import load_pil_images 

        vl_chat_processor = DeepseekVLV2Processor.from_pretrained(
            model_path,
            cache_dir=args.model_para_path
        )
        vl_gpt: DeepseekVLV2ForCausalLM = AutoModelForCausalLM.from_pretrained(
            model_path,
            # device_map="auto",
            trust_remote_code=True,
            cache_dir=args.model_para_path
        )

    # Extract tokenizer from processor
    tokenizer = vl_chat_processor.tokenizer
    
    try:
        if args.eval_model == "DeepSeek-VL-7B-Chat":
            # Configure model for inference
            vl_gpt = vl_gpt.to(torch.bfloat16).cuda().eval()
        else:
            vl_gpt = vl_gpt.to(torch.bfloat16).cuda().eval()   # .cuda()
            if args.use_deepspeed:
                vl_gpt = deepspeed.init_inference(
                    vl_gpt,
                    tensor_parallel={"tp_size": 4},
                    dtype=torch.bfloat16,
                    # replace_with_kernel_inject=True
                    ).module

    except RuntimeError as e:
        raise ValueError(f"Failed to move model to CUDA: {str(e)}")

    return vl_chat_processor, tokenizer, vl_gpt

def run_model(args):
    """
    Run the DeepSeek-VL series model evaluation.
    
    Args:
        args: Argument parser containing configuration parameters.
    """
    model_path = args.module

    # Load model and processor
    vl_chat_processor, tokenizer, vl_gpt = load_model_and_processor(args, model_path)

    # Validate batch size
    if args.batchsize > 1:
        logging.error(
            "DeepSeek-VL-7B does not support batched processing of independent image-text pairs. Please set batchsize to 1."
            )
        raise ValueError("Invalid batch size: DeepSeek-VL-7B requires batchsize to be 1.")

    torch.cuda.empty_cache()

    # Initialize dataset and dataloader
    dataset = DeepSeekVL7BChatDataset(args, vl_chat_processor, vl_gpt)
    # Set sampler and DataLoader based on DDP mode
    sampler = torch.utils.data.distributed.DistributedSampler(dataset, shuffle=False) if args.ddp else None
    data_loader = torch.utils.data.DataLoader(
        dataset,
        batch_size=args.batchsize,
        sampler=sampler,
        shuffle=(sampler is None),  # Only shuffle if not using DDP
        collate_fn=deepseek_vl_collate_fn
    )

    # Enable Distributed Data Parallel (DDP)
    with torch.no_grad():
        vl_gpt = torch.nn.parallel.DistributedDataParallel(vl_gpt, device_ids=[args.local_rank]) if args.ddp else vl_gpt

    
    start_time = time.time()



    all_data = {}

    for idx, batch_data in enumerate(tqdm(data_loader)):

        # Load batch data
        img_id                              = batch_data.get('img_id')[0]
        img_path                            = batch_data.get('img_path')[0]
        caption                             = batch_data.get('caption')
        object_class_QA                     = batch_data.get('object_class_QA')[0]
        object_colour_QA                    = batch_data.get('object_colour_QA')[0]
        object_num_QA                       = batch_data.get('object_num_QA')[0]
        is_counterintuitive_QA              = batch_data.get('is_counterintuitive_QA')[0]
        relation_QA                         = batch_data.get('relation_QA')[0]
        Anomaly_Recognition_QA              = batch_data.get('Anomaly_Recognition_QA')[0]
        caption_instruction                 = args.caption_instruction


        # Data format & Model inference
        # class question & model inference
        object_class_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, object_class_QA, img_path)
        object_class_answer = deepseekvl_forward(args, vl_gpt, tokenizer, object_class_input)

        # colour question & model inference
        object_colour_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, object_colour_QA, img_path)
        object_colour_answer = deepseekvl_forward(args, vl_gpt, tokenizer, object_colour_input)

        # num question & model inference
        object_num_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, object_num_QA, img_path)
        object_num_answer = deepseekvl_forward(args, vl_gpt, tokenizer, object_num_input)

        # counterintuitive question & model inference
        is_counterintuitive_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, is_counterintuitive_QA, img_path)
        counterintuitive_answer = deepseekvl_forward(args, vl_gpt, tokenizer, is_counterintuitive_input)

        # relation question & model inference
        relation_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, relation_QA, img_path)
        relation_answer = deepseekvl_forward(args, vl_gpt, tokenizer, relation_input)

        # anomaly recognition question & model inference
        Anomaly_Recognition_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, Anomaly_Recognition_QA, img_path)
        Anomaly_Recognition_answer = deepseekvl_forward(args, vl_gpt, tokenizer, Anomaly_Recognition_input)

        # Image caption & model inference
        caption_instruction_input = deepseekvl_format_data(args, vl_chat_processor, vl_gpt, caption_instruction, img_path)
        caption_instruction_answer = deepseekvl_forward(args, vl_gpt, tokenizer, caption_instruction_input)

        answer_label_dict = ContraMetric(args, object_class_answer, object_class_QA, object_colour_answer, object_colour_QA, object_num_answer, object_num_QA, counterintuitive_answer, is_counterintuitive_QA, relation_answer, relation_QA, Anomaly_Recognition_answer, Anomaly_Recognition_QA)

        answer_label_dict["caption_answer"] = caption_instruction_answer
        answer_label_dict["caption_label"] = caption

        all_data[img_id] = answer_label_dict

    end_time = time.time()
    formatted_time = format_elapsed_time(start_time, end_time)

    # Final evaluation metrics
    if args.local_rank == 0 or not args.ddp:
        logging.info(f"-Formatted time: {formatted_time}, GPUs num is {args.nproc_per_node}. The dataset has a total of {len(dataset)} samples complete evaluation of the performance of the metrics as follows: - Contra Acc: {args.contra_metric_correct_num / len(dataset)}. Correct num: {args.contra_metric_correct_num}. Total num: {len(dataset)}\n")
        logging.info(all_data)
